<?php

namespace App\Filament\Pages\Auth;

use App\Models\User;
use DanHarrin\LivewireRateLimiting\Exceptions\TooManyRequestsException;
use Filament\Facades\Filament;
use Filament\Http\Responses\Auth\Contracts\LoginResponse;
use Filament\Models\Contracts\FilamentUser;
use Filament\Notifications\Notification;
use Filament\Pages\Auth\Login as BaseLogin;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\HtmlString;

class CustomLogin extends BaseLogin
{

    /**
     * @var view-string
     */
    protected static string $view = 'filament.auth.login';


    protected function getForms(): array
    {
        return [
            'form' => $this->form(
                $this->makeForm()
                    ->schema([
                        $this->getEmailFormComponent()->label(__('messages.mails.email_address') . ':')->validationAttribute(__('messages.mails.email_address'))->placeholder(__('messages.mails.email_address')),
                        $this->getPasswordFormComponent()->label(__('messages.staff.password') . ':')->validationAttribute(__('messages.staff.password'))->placeholder(__('messages.staff.password'))
                            ->hint(filament()->hasPasswordReset() ? new HtmlString(Blade::render('<x-filament::link :href="filament()->getRequestPasswordResetUrl()" tabindex="3"> {{ __("messages.forgot_password") }}</x-filament::link>')) : null),
                        $this->getRememberFormComponent()->label(__('messages.common.remember_me')),
                    ])
                    ->statePath('data'),
            ),
        ];
    }

    protected function getFormActions(): array
    {
        return [
            $this->getAuthenticateFormAction()
                ->extraAttributes(['class' => 'w-full flex items-center justify-center space-x-3'])
                ->label(__('messages.common.sign_in')),
        ];
    }


    public function authenticate(): ?LoginResponse
    {
        try {
            $this->rateLimit(5);
        } catch (TooManyRequestsException $exception) {
            $this->getRateLimitedNotification($exception)?->send();

            return null;
        }

        $data = $this->form->getState();

        if (isset($data['email']) && !empty($data['email'])) {
            $user = User::whereEmail($data['email'])->first();
            if ($user) {
                if ($user->email_verified_at == null) {
                    Notification::make()
                        ->title(__('messages.placeholder.email_not_verified'))
                        ->danger()
                        ->send();

                    return null;
                }
                if ($user->status == false) {
                    Notification::make()
                        ->title(__('messages.placeholder.your_account_is_currently_disabled_please_contact_to_administrator'))
                        ->danger()
                        ->send();

                    return null;
                }
            }
        }

        if (! Filament::auth()->attempt($this->getCredentialsFromFormData($data), $data['remember'] ?? false)) {
            $this->throwFailureValidationException();
        }

        $user = Filament::auth()->user();

        if (
            ($user instanceof FilamentUser) &&
            (! $user->canAccessPanel(Filament::getCurrentPanel()))
        ) {
            Filament::auth()->logout();

            $this->throwFailureValidationException();
        }

        session()->regenerate();

        return app(LoginResponse::class);
    }
}
