<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;

use App\Notifications\UserEmailVerification;
use Filament\Forms\Components\Fieldset;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Radio;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Models\Contracts\FilamentUser;
use Filament\Models\Contracts\HasAvatar;
use Filament\Models\Contracts\HasName;
use Filament\Panel;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Spatie\Permission\Traits\HasRoles;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Lang;

class User extends Authenticatable implements FilamentUser, HasName, HasMedia, MustVerifyEmail, HasAvatar
{
    use HasFactory, Notifiable, InteractsWithMedia, HasRoles;

    protected $table = 'users';

    public function canAccessPanel(Panel $panel): bool
    {
        return true;
    }

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'first_name',
        'last_name',
        'email',
        'contact',
        'dob',
        'gender',
        'status',
        'password',
        'language',
        'dark_mode',
        'blood_group',
        'type',
        'email_verified_at',
        'about_us',
        'username',
        'last_seen_at',
        'is_default',
    ];

    protected $casts = [
        'first_name' => 'string',
        'last_name' => 'string',
        'email' => 'string',
        'password' => 'hashed',
        'dob' => 'date',
        'contact' => 'string',
        'gender' => 'integer',
        'status' => 'integer',
        'dark_mode' => 'integer',
        'type' => 'integer',
        'remember_token' => 'string',
        'language' => 'string',
        'blood_group' => 'string',
    ];

    const PROFILE = 'profile';

    const COVER_IMG = 'cover_img';

    const NEWS_IMAGE = 'news-image';

    const ADMIN = 1;

    const STAFF = 2;

    const TYPE = [
        self::ADMIN => 'Admin',
        self::STAFF => 'Staff',
    ];

    protected $with = ['media'];

    protected $appends = ['full_name', 'profile_image', 'role_name', 'cover_image'];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    const MALE = 1;

    const FEMALE = 2;

    const GENDER = [
        self::MALE => 'Male',
        self::FEMALE => 'Female',
    ];

    public static $rules = [
        'first_name' => 'required|max:190',
        'last_name' => 'required|max:190',
        'email' => 'required|max:160|email:filter|unique:users,email',
        'password' => 'required|same:password_confirmation|min:6|max:190',
        'dob' => 'nullable|date',
        'contact' => 'required|numeric',
        'experience' => 'nullable|numeric',
        'specializations' => 'required',
        'gender' => 'required',
        'status' => 'nullable',
    ];

    public static function getGenderType(): array
    {
        $genderType[self::MALE] = Lang::get('messages.staff.' . self::MALE);

        $genderType[self::FEMALE] = Lang::get('messages.staff.' . self::FEMALE);

        return $genderType ?? [];
    }

    public function getProfileImageAttribute(): string
    {
        /** @var Media $media */
        $media = $this->getMedia(self::PROFILE)->first();
        if (! empty($media)) {
            return $media->getFullUrl();
        }

        return asset('assets/image/avatar.png');
    }

    public function getProfileUrlAttribute()
    {
        if ($this->hasMedia(self::PROFILE)) {
            return $this->getFirstMediaUrl(self::PROFILE);
        }

        return asset('assets/image/avatar.png');
    }

    public function getCoverImageAttribute()
    {
        /** @var Media $media */
        $media = $this->getMedia(self::COVER_IMG)->first();
        if (! empty($media)) {
            return $media->getFullUrl();
        }

        return asset('assets/image/post-image/post-17.jpg');
    }

    public function getFilamentAvatarUrl(): ?string
    {
        return $this->getFirstMediaUrl(self::PROFILE);
    }

    public function getRoleNameAttribute()
    {
        $role = $this->roles()->first();

        if (! empty($role)) {
            return $role->display_name;
        }
    }

    public function getFilamentName(): string
    {
        return "{$this->first_name} {$this->last_name}";
    }

    public function getFullNameAttribute()
    {
        return ucfirst($this->first_name) . ' ' . ucfirst($this->last_name);
    }

    public function staff(): HasOne
    {
        return $this->hasOne(Staff::class);
    }

    public function subscription(): BelongsTo
    {
        return $this->belongsTo(Subscription::class, 'id', 'user_id')
            ->where('status', Subscription::ACTIVE);
    }

    // public function canAccessPanel(Panel $panel): bool
    // {
    //     return $this->hasRole('admin|staff|customer');
    //     // return true;
    // }

    public function sendEmailVerificationNotification()
    {
        $mailData = MailSetting::first();
        $protocol = MailSetting::TYPE[$mailData->mail_protocol];
        $host = $mailData->mail_host;

        if ($mailData->mail_protocol == MailSetting::MAIL_LOG) {
            $protocol = 'log';
            $host = 'mailhog';
        }

        if ($mailData->mail_protocol == MailSetting::SMTP) {
            $protocol = 'smtp';
        }

        if ($mailData->mail_protocol == MailSetting::SENDGRID) {
            $protocol = 'sendgrid';
        }

        config(
            [
                'mail.default' => $protocol,
                "mail.mailers.$protocol.transport" => $protocol,
                "mail.mailers.$protocol.host" => $host,
                "mail.mailers.$protocol.port" => $mailData->mail_port,
                "mail.mailers.$protocol.encryption" => MailSetting::ENCRYPTION_TYPE[$mailData->encryption],
                "mail.mailers.$protocol.username" => $mailData->mail_username,
                "mail.mailers.$protocol.password" => $mailData->mail_password,
                'mail.from.address' => $mailData->reply_to,
                'mail.from.name' => $mailData->mail_title,
            ]
        );
        $this->notify(new UserEmailVerification);
    }

    public static function getForm($form)
    {
        return [
            Section::make()
                ->schema([
                    TextInput::make('first_name')
                        ->required()
                        ->label(__('messages.staff.first_name') . ':')
                        ->validationAttribute(__('messages.staff.first_name'))
                        ->placeholder(__('messages.staff.first_name')),
                    TextInput::make('last_name')
                        ->required()
                        ->label(__('messages.staff.last_name') . ':')
                        ->validationAttribute(__('messages.staff.last_name'))
                        ->placeholder(__('messages.staff.last_name')),
                    TextInput::make('email')
                        ->required()
                        ->label(__('messages.staff.email') . ':')
                        ->validationAttribute(__('messages.staff.email'))
                        ->placeholder(__('messages.staff.email'))
                        ->email()
                        ->unique(ignorable: fn(?User $record) => $record),
                    TextInput::make('contact')
                        ->required()
                        ->tel()
                        ->rules(['required', 'regex:/^[0-9]{10}$/'])
                        ->label(__('messages.staff.contact_no') . ':')
                        ->validationAttribute(__('messages.staff.contact_no'))
                        ->placeholder(__('messages.staff.contact_no')),
                    TextInput::make('username')
                        ->required()
                        ->label(__('messages.staff.username') . ':')
                        ->validationAttribute(__('messages.staff.username'))
                        ->placeholder(__('messages.staff.username'))
                        ->autocomplete(false)
                        ->unique(ignorable: fn(?User $record) => $record),
                    Select::make('roles')
                        ->label(__('messages.staff.role') . ':')
                        ->validationAttribute(__('messages.staff.role'))
                        ->placeholder(__('messages.staff.role'))
                        ->required()
                        ->searchable()
                        ->options(Role::whereNotIn('name', ['customer'])->pluck('name', 'id'))
                        ->saveRelationshipsUsing(function (Model $record, $state) {
                            $record->roles()->sync($state);
                        })
                        ->native(false)
                        ->preload()
                        ->hidden($form->getOperation() == 'edit'),
                    TextInput::make('password')
                        ->label(__('messages.staff.password') . ':')
                        ->validationAttribute(__('messages.staff.password'))
                        ->placeholder(__('messages.staff.password'))
                        ->password()
                        ->required()
                        ->revealable()
                        ->autocomplete(false)
                        ->maxLength(255)
                        ->rules(['min:8'])
                        ->visible(function (?string $operation) {
                            return $operation == 'create';
                        }),
                    TextInput::make('password_confirmation')
                        ->label(__('messages.user.confirm_password') . ':')
                        ->validationAttribute(__('messages.user.confirm_password'))
                        ->placeholder(__('messages.user.confirm_password'))
                        ->password()
                        ->required()
                        ->revealable()
                        ->same('password')
                        ->maxLength(255)
                        ->rules(['min:8'])
                        ->visible(function (?string $operation) {
                            return $operation == 'create';
                        }),

                    Fieldset::make('Label')
                        ->label('')
                        ->schema([
                            Radio::make('gender')
                                ->label(__('messages.staff.gender') . ':')
                                ->validationAttribute(__('messages.staff.gender'))
                                ->columns(2)
                                ->required()
                                ->options(User::getGenderType())
                                ->afterStateHydrated(function ($state, $set, $operation) {
                                    if ($operation == 'edit' && (empty($state) || $state == 0)) {
                                        $set('gender', User::MALE);
                                    }
                                })
                                ->default(User::MALE),
                            Toggle::make('status')
                                ->label(__('messages.status') . ':')
                                ->validationAttribute(__('messages.status'))
                                ->inline(false)
                                ->default(true),
                        ])->columns(2)->columnSpan(1),

                    Textarea::make('about_us')
                        ->rows(3)
                        ->label(__('messages.staff.about_us') . ':')
                        ->validationAttribute(__('messages.staff.about_us'))
                        ->placeholder(__('messages.staff.about_us'))
                        ->maxLength(255),
                    SpatieMediaLibraryFileUpload::make('profile')
                        ->label(__('messages.staff.profile') . ':')
                        ->reorderable()
                        ->disk(config('app.media_disk'))
                        ->collection(Staff::PROFILE),
                    SpatieMediaLibraryFileUpload::make('cover_img')
                        ->label(__('messages.staff.cover_image') . ':')
                        ->reorderable()
                        ->disk(config('app.media_disk'))
                        ->collection(Staff::COVER_IMG),
                    Hidden::make('type')->default(User::STAFF),
                ])->columns(2)
        ];
    }
}
