<?php

namespace App\Providers;

use App\Models\Language;
use Filament\Support\Assets\Js;
use Illuminate\Support\ServiceProvider;
use Filament\Support\Facades\FilamentAsset;
use BezhanSalleh\FilamentLanguageSwitch\LanguageSwitch;
use BezhanSalleh\FilamentLanguageSwitch\Enums\Placement;
use Filament\Http\Responses\Auth\Contracts\LoginResponse;
use Filament\Http\Responses\Auth\Contracts\LogoutResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Mariuzzo\LaravelJsLocalization\Commands\LangJsCommand;
use Mariuzzo\LaravelJsLocalization\Generators\LangJsGenerator;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        $this->app->singleton(
            LoginResponse::class,
            \App\Http\Responses\LoginResponse::class
        );

        $this->app->singleton(
            LogoutResponse::class,
            \App\Http\Responses\LogoutResponse::class
        );

        // Bind the Laravel JS Localization command into the app IOC.
        $this->app->singleton('localization.js', function ($app) {
            $app = $this->app;
            $laravelMajorVersion = (int) $app::VERSION;

            $files = $app['files'];

            if ($laravelMajorVersion === 4) {
                $langs = $app['path.base'] . '/app/lang';
            } elseif ($laravelMajorVersion >= 5 && $laravelMajorVersion < 9) {
                $langs = $app['path.base'] . '/resources/lang';
            } elseif ($laravelMajorVersion >= 9) {
                $langs = app()->langPath();
            }
            $messages = $app['config']->get('localization-js.messages');
            $generator = new LangJsGenerator($files, $langs, $messages);

            return new LangJsCommand($generator);
        });
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        if ($this->isDbConnected()) {
            LanguageSwitch::configureUsing(function (LanguageSwitch $switch) {
                $languages =  Language::pluck('iso_code')->toArray();


                $switch
                    // ->locales(['en','ar','zh'])
                    ->locales($languages)
                    ->flags([
                        'ar' => asset('images/flags/arabic.svg'),
                        'en' => asset('images/flags/english.png'),
                        'fr' => asset('images/flags/france.png'),
                        'de' => asset('images/flags/german.png'),
                        'es' => asset('images/flags/spain.png'),
                        'pt' => asset('images/flags/portuguese.png'),
                        'it' => asset('images/flags/italian.png'),
                        'ru' => asset('images/flags/russian.png'),
                        'tr' => asset('images/flags/turkish.png'),
                        'zh' => asset('images/flags/china.png'),
                    ])
                    ->outsidePanelPlacement(Placement::TopLeft)
                    ->visible(outsidePanels: true)
                    ->outsidePanelRoutes([
                        'auth.login',
                        'auth.register',
                        'auth.password-reset',
                    ]);
            });


            FilamentAsset::register([
                // Js::make('custom-script', __DIR__.'/../../resources/js/custom.js'),
            ]);
            // PLZ run this command in terminal to publish custom js: php artisan filament:assets
            // \URL::forceScheme('https');
        } else {
            // Handle the case where the database is not connected
            Log::error('Database connection failed.');
        }
    }

    protected function isDbConnected(): bool
    {
        try {
            DB::connection()->getPdo();
            return true;
        } catch (\Exception $e) {
            Log::error('Database connection error: ' . $e->getMessage());
            return false;
        }
    }
}
